import logging
import os
import sys
import yaml
from logging.handlers import RotatingFileHandler
from typing import Dict, Any

_log_handlers = set()

def load_yaml_config(yaml_path: str) -> Dict[str, Any]:
    """
        Load and parse the YAML configuration file.
    """
    try:
        with open(yaml_path, 'r', encoding='utf-8') as file:
            return yaml.load(file, Loader=yaml.SafeLoader)
    except FileNotFoundError:
        raise Exception(f"Error: Configuration file '{yaml_path}' not found.")
    except yaml.YAMLError as e:
        raise Exception(f"Error parsing YAML file: {e}")

def setup_logging(log_file: str = "pipeline.log", log_level=logging.INFO):
    """Configures root logger. Ensures handlers are added only once."""
    global _log_handlers
    log_format = '%(asctime)s - %(process)d - %(name)s - %(levelname)s - %(message)s'
    date_format = '%Y-%m-%d %H:%M:%S'

    log_dir = os.path.join("Files", "logs")
    os.makedirs(log_dir, exist_ok=True)
    log_path = os.path.join(log_dir, log_file)
    
    logger = logging.getLogger()
    logger.setLevel(log_level)

    handler_key_file = f"file_{log_path}"
    if handler_key_file not in _log_handlers:
        file_handler = RotatingFileHandler(
            log_path,
            maxBytes=100*1024*1024,
            backupCount=5,
            encoding='utf-8'
        )
        file_handler.setFormatter(logging.Formatter(log_format, date_format))
        logger.addHandler(file_handler)
        _log_handlers.add(handler_key_file)
        
    handler_key_console = "console_stdout"
    if handler_key_console not in _log_handlers:
        console_handler = logging.StreamHandler(sys.stdout)
        console_handler.setFormatter(logging.Formatter(log_format, date_format))
        logger.addHandler(console_handler)
        _log_handlers.add(handler_key_console)
    return logger